// Adobe.FaceTracker.js
/*global define, console */
/*jslint sub: true */

define([ "lib/Zoot", "src/utils", "src/math/Vec2", "src/math/mathUtils", "lib/dev", "lib/tasks"],
  function (Zoot, utils, vec2, mathUtils, dev, tasks) {
		"use strict";

	var V2 = Zoot.Vec2, // everyN = 0,
		pKeyCode = Zoot.keyCodes.getKeyGraphId(";"),	// This should be a zstring, but how do we translate it here?  -jacquave

		// see https://bitbucket.org/amitibo/pyfacetracker/src/d54866d9b3e23654b1c06adca625dafcbe7629ce/doc/images/3DMonaLisa.png?at=default
		//	for diagram of these indices
//		faceFeatureIndices = {
//			"Head" :					[0, 16], // jaw line going from puppet right to left
//			"Head/Right Eye" :			[36, 41],
//			"Head/Left Eye" :			 [42, 47],
//			"Head/Right Eyebrow" :		 [17, 21],
//			"Head/Left Eyebrow" :		 [22, 26],
//			"Head/Nose" :				 [27, 35], // 27-30 down ridge, 31-35 right to left
//			"Head/Mouth" :				 [48, 65]
//		},


//		note: mouths are in UI order
// 		hard-coded, instead of sorting by UI name, so the order remains the same across UI languages
 
		mouthShapeLayerTagDefinitions = [
			{
				id: "Adobe.Face.NeutralMouth",	// see parallel definition in FaceTracker
				artMatches: ["neutral"],
				uiName: "$$$/animal/Behavior/Face/TagName/Neutral=Neutral",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Mouth", subSort:1}]				
			},
			{
				id: "Adobe.Face.SmileMouth",
				artMatches: ["smile"],
				uiName: "$$$/animal/Behavior/Face/TagName/Smile=Smile",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Mouth", subSort:1}]				
			},
			{
				id: "Adobe.Face.SurprisedMouth",
				artMatches: ["surprised"],
				uiName: "$$$/animal/Behavior/Face/TagName/Surprised=Surprised",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Mouth", subSort:1}]				
			},
		],
		
		mouthParentLayerTagDefinition = [
			{
				id: "Adobe.Face.MouthsParent",
				artMatches: ["mouth"],
				uiName: "$$$/animal/Behavior/Face/TagName/MouthGroup=Mouth Group",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Face"}]				
			},
		],
		
		viewLayerTagDefinitions = [
			{
				id: "Adobe.Face.LeftProfile",
				artMatches: ["left profile"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftProfile=Left Profile",
			},
			
			{
				id: "Adobe.Face.LeftQuarter",
				artMatches: ["left quarter"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftQuarter=Left Quarter",
			},
			
			{
				id: "Adobe.Face.Front",
				artMatches: ["frontal"],
				uiName: "$$$/animal/Behavior/Face/TagName/Frontal=Frontal",
			},
			
			{
				id: "Adobe.Face.RightQuarter",
				artMatches: ["right quarter"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightQuarter=Right Quarter",
			},
			
			{
				id: "Adobe.Face.RightProfile",
				artMatches: ["right profile"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightProfile=Right Profile",
			},
			
			{
				id: "Adobe.Face.Upward",
				artMatches: ["upward"],
				uiName: "$$$/animal/Behavior/Face/TagName/Upward=Upward",
			},

			{
				id: "Adobe.Face.Downward",
				artMatches: ["downward"],
				uiName: "$$$/animal/Behavior/Face/TagName/Downward=Downward",
			},
			
		];
		
		viewLayerTagDefinitions.forEach (function (tagDefn) {
			tagDefn.tagType = "layertag";
			tagDefn.uiGroups = [{ id:"Adobe.TagGroup.HeadTurn"}];			
		});
		
		var miscLayerTagDefinitions = [
			{
				id: "Adobe.Face.LeftBlink",
				artMatches: ["left blink"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftBlink=Left Blink",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Face"}]				

			},
			{
				id: "Adobe.Face.RightBlink",
				artMatches: ["right blink"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightBlink=Right Blink",
				tagType: "layertag",
				uiGroups: [{ id:"Adobe.TagGroup.Face"}]								
			},
		],

		faceLayerTagDefinitions = [
			{
				id: "Adobe.Face.LeftPupilRange",
				artMatches: ["left eyeball", "left pupil range"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftPupilRange=Left Pupil Range",
			},
		
			{
				id: "Adobe.Face.RightPupilRange",
				artMatches: ["right eyeball", "right pupil range"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightPupilRange=Right Pupil Range",
			},

			{
				id: "Adobe.Face.LeftPupilSize",
				artMatches: ["left pupil", "left pupil size"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftPupilSize=Left Pupil Size",
			},
		
			{
				id: "Adobe.Face.RightPupilSize",
				artMatches: ["right pupil", "right pupil size"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightPupilSize=Right Pupil Size",
			},
			
			{
				id: "Adobe.Face.LeftEyelidTopLayer",
				artMatches: ["left eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidTopSize=Left Eyelid Top Size",
			},
		
			{
				id: "Adobe.Face.LeftEyelidBottomLayer",
				artMatches: ["left eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidBottomSize=Left Eyelid Bottom Size",
			},
		
			{
				id: "Adobe.Face.RightEyelidTopLayer",
				artMatches: ["right eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidTopSize=Right Eyelid Top Size",
			},
		
			{
				id: "Adobe.Face.RightEyelidBottomLayer",
				artMatches: ["right eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidBottomSize=Right Eyelid Bottom Size",
			},
		];
		
		faceLayerTagDefinitions.forEach ( function (tagDefn) {
			tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];							
			tagDefn.tagType = "layertag";
		});
		// all possible face features presented as tag definitions, also used in defineHandleParams()

		var faceFeatureTagDefinitions = [	
		
			{
				id: "Adobe.Face.Head",
				artMatches: ["head"],
				uiName: "$$$/animal/Behavior/Face/TagName/Head=Head",
			},
		
			{
				id: "Adobe.Face.RightEye",
				artMatches: ["right eye"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEye=Right Eye",
			},
		
			{
				id: "Adobe.Face.RightEyelidTop",
				artMatches: ["right eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidTop=Right Eyelid Top",
			},
		
			{
				id: "Adobe.Face.RightEyelidBottom",
				artMatches: ["right eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyelidBottom=Right Eyelid Bottom",
			},
		
			{
				id: "Adobe.Face.RightPupil",
				artMatches: ["right pupil"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightPupil=Right Pupil",
			},
		
			{
				id: "Adobe.Face.LeftEye",
				artMatches: ["left eye"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEye=Left Eye",
			},
		
			{
				id: "Adobe.Face.LeftEyelidTop",
				artMatches: ["left eyelid top"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidTop=Left Eyelid Top",
			},
		
			{
				id: "Adobe.Face.LeftEyelidBottom",
				artMatches: ["left eyelid bottom"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyelidBottom=Left Eyelid Bottom",
			},
		
			{
				id: "Adobe.Face.LeftPupil",
				artMatches: ["left pupil"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftPupil=Left Pupil",
			},
		
			//"Adobe.Face.Right Blink",		don't need these because there are no transforms related to them (just replacements)
			//"Adobe.Face.Left Blink",

			{
				id: "Adobe.Face.RightEyebrow",
				artMatches: ["right eyebrow"],
				uiName: "$$$/animal/Behavior/Face/TagName/RightEyebrow=Right Eyebrow",
			},
		
			{
				id: "Adobe.Face.LeftEyebrow",
				artMatches: ["left eyebrow"],
				uiName: "$$$/animal/Behavior/Face/TagName/LeftEyebrow=Left Eyebrow",
			},
		
			{
				id: "Adobe.Face.Nose",
				artMatches: ["nose"],
				uiName: "$$$/animal/Behavior/Face/TagName/Nose=Nose",
			},
		
			{
				id: "Adobe.Face.Mouth",
				artMatches: ["mouth"],
				uiName: "$$$/animal/Behavior/Face/TagName/Mouth=Mouth",
			}
		];
		
		faceFeatureTagDefinitions.forEach( function (tagDefn) {
			tagDefn.tagType = "handletag";
			tagDefn.uiGroups = [{ id:"Adobe.TagGroup.Face"}];					
		});

		
		var head14Labels = { // note: values are identity; repeated in .cpp code in FaceMetrics constructor (TODO: factor)
			"Head/DX" : 0,
			"Head/DY" : 0,
			"Head/DZ" : 0,
			"Head/Orient/X" : 0,
			"Head/Orient/Y" : 0,
			"Head/Orient/Z" : 0,
			"Head/LeftEyebrow" : 1,
			"Head/RightEyebrow" : 1,
			"Head/LeftEyelid" : 1,
			"Head/RightEyelid" : 1,
			"Head/MouthSX" : 1,
			"Head/MouthSY" : 1,
			"Head/MouthDX" : 0,
			"Head/MouthDY" : 0,
			"Head/Scale" : 1,
			"Head/MouthShape" : -1,
			"Head/LeftEyeGazeX" : 0,
			"Head/LeftEyeGazeY" : 0, 
			"Head/RightEyeGazeX" : 0, 
			"Head/RightEyeGazeY" : 0
		},

		// first three are default mouth shapes (match default mouth shape classifiers in beaker::zoot:FaceClassifiers)
		// TODO: eventually allow users to add to this list with appropriate named mouth shapes for customized facial expression recognition
		mouthShapeLabels = [
			"Adobe.Face.NeutralMouth",
			"Adobe.Face.SurprisedMouth",		// warning: the indexes of these must match FaceClassifiers::GetMouthShape2DSimple()
			"Adobe.Face.SmileMouth"
		],

		// sequences of fall back mouth shape indices
		// whichever mouth shape index the facetracker returns, 
		// chooseMouthReplacements will use the first valid mouth shape in the corresponding sequence
		mouthShapeCascades = {
			0: [0], 
			1: [1, 0],
			2: [2, 0]
			
//			3: [3, 1, 0], commented out along with "Grimace" - "Tongue" above
//			4: [4, 0],
		},

		// stores mapping from head14 params to puppet transformations	
		// all transformation values are unitless; when transforms are applied
		// at runtime, they get multiplied by the appropriate puppet-specific measurements
		// note: no mappings for Left/Right Blink because it's only hidden or shown
		head14ToPuppetTransformMapping = { 
			"Adobe.Face.Head" :
			[
			{
				"labels" : ["Head/DX"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/DX" : -2 
					},
					"T" : {
						"translate" : [-2, 0], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/DX" : 2
					},
					"T" : {
						"translate" : [2, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			},
			{
				"labels" : ["Head/DY"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/DY" : -2 
					},
					"T" : {
						"translate" : [0, -2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/DY" : 2
					},
					"T" : {
						"translate" : [0, 2],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			},
			{
				"labels" : ["Head/Orient/Z"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/Orient/Z" : -1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1],
						"angle" : -1.5
					}
				},
				{
					"head14" : {
						"Head/Orient/Z" : 1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1], 
						"angle" : 1.5
					}
				}
				]
			}			
			],

			"Adobe.Face.Nose" : 
			[
			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "noseDepth", 
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]);
					T.translate[1] += 0.5 * Math.sin(x[1]); 
				}
			}
			],

			"Adobe.Face.LeftEyebrow" :
			[
			{
				"labels" : ["Head/LeftEyebrow"],
				"translateUnit" : "leftEyeEyebrowDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/LeftEyebrow" : 0.5
					},
					"T" : {
						"translate" : [0, 2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/LeftEyebrow" : 2
					},
					"T" : {
						"translate" : [0, -4], 
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}

			],

			"Adobe.Face.RightEyebrow" :
			[
			{
				"labels" : ["Head/RightEyebrow"],
				"translateUnit" : "rightEyeEyebrowDist",
				"samples" : [

				{
					"head14" : {
						"Head/RightEyebrow" : 0.5
					},
					"T" : {
						"translate" : [0, 2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyebrow" : 2
					},
					"T" : {
						"translate" : [0, -4],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}

			],

			"Adobe.Face.RightEye" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1.2], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}
			],

			"Adobe.Face.RightEyelidTop" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"translateUnit" : "rightEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.RightEyelidBottom" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"translateUnit" : "rightEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.RightPupil" :
			[
			{
				"labels" : ["Head/RightEyeGazeX"],
				"translateUnit" : "rightEyeWidth",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyeGazeX" : -0.8
					},
					"T" : {
						"translate" : [-0.45, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyeGazeX" : 0.8
					},
					"T" : {
						"translate" : [0.45, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/RightEyeGazeY"],
				"translateUnit" : "rightEyeHeight",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyeGazeY" : -0.8
					},
					"T" : {
						"translate" : [0, -0.3],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyeGazeY" : 0.6
					},
					"T" : {
						"translate" : [0, 0.225],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.LeftEye" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1.2], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}
			],

			"Adobe.Face.LeftEyelidTop" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"translateUnit" : "leftEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.LeftEyelidBottom" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"translateUnit" : "leftEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			}
			],

			"Adobe.Face.LeftPupil" :
			[
			{
				"labels" : ["Head/LeftEyeGazeX"],
				"translateUnit" : "leftEyeWidth",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyeGazeX" : -0.8
					},
					"T" : {
						"translate" : [-0.45, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyeGazeX" : 0.8
					},
					"T" : {
						"translate" : [0.45, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/LeftEyeGazeY"],
				"translateUnit" : "leftEyeHeight",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyeGazeY" : -0.8
					},
					"T" : {
						"translate" : [0, -0.3],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyeGazeY" : 0.6
					},
					"T" : {
						"translate" : [0, 0.225],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Adobe.Face.Mouth" :
			[
			{
				"labels" : ["Head/MouthDX"],
				"translateUnit" : "mouthWidth",				
				"samples" : [

				{
					"head14" : {
						"Head/MouthDX" : -1
					},
					"T" : {
						"translate" : [-1, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthDX" : 1
					},
					"T" : {
						"translate" : [1, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				}

				]
			},
			{
				"labels" : ["Head/MouthDY"],
				"translateUnit" : "mouthHeight",				
				"samples" : [

				{
					"head14" : {
						"Head/MouthDY" : -1
					},
					"T" : {
						"translate" : [0, -1],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthDY" : 1
					},
					"T" : {
						"translate" : [0, 1],
						"scale" : [1, 1],
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/MouthSX"],
				"samples" : [

				{
					"head14" : {
						"Head/MouthSX" : 0
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [0, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthSX" : 1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1.5, 1],
						"angle" : 0
					}
				}

				]
			},
			{
				"labels" : ["Head/MouthSY"],
				"samples" : [

				{
					"head14" : {
						"Head/MouthSY" : 0
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthSY" : 5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 5],
						"angle" : 0
					}
				}
				]
			}, 
			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}			
			]
		};
	// end of var declarations
	
			
	function getHead14(args) {
		var label, v, vals = {}, inputId = "cameraInput";

		// use first label existence as a proxy for all existing
		if (args.getParamEventValue(inputId, "Head/InputEnabled", null, false)) {
			args.setEventGraphParamRecordingValid(inputId);
			for (label in head14Labels) {
				if (head14Labels.hasOwnProperty(label)) {
				    v = args.getParamEventValue(inputId, label, null, head14Labels[label]);
					if (v !== undefined) {
						vals[label] = v;
					} else { // else  missing part of head14 data 
						vals[label] = head14Labels[label];
					}
				}
			}
		} else {
			// return identity head14 -- perhaps camera tracking not spun up yet
			for (label in head14Labels) {
				if (head14Labels.hasOwnProperty(label)) {
					vals[label] = head14Labels[label];
				}
			}
		}
		
		return vals;
	}

	function getFilteredHead14(args) {
		var label, v, vals = {}, inputId = "cameraInput", k = args.getParamEventOutputKey(inputId), labelKey;

		// use first label existence as a proxy for all existing
		for (label in head14Labels) {
			if (head14Labels.hasOwnProperty(label)) {
				labelKey = k + "Filtered/Head/" + label;
				if (canIterpolateHeadLabel(label)) {
					v = args.getParamEventValue(inputId, labelKey, null, head14Labels[label]);
				} else {
					var graphEvaluatorArray = args.getGraphEvaluatorArray(inputId);
					if (graphEvaluatorArray.length) {
						v = args.getParamEventValueFromEvaluator(graphEvaluatorArray[0], inputId, labelKey);
					} else {
						v = undefined;
					}
				}
				
				if (v !== undefined) {
					vals[label] = v;
				} else { // else  missing part of head14 data 
					vals[label] = head14Labels[label];
				}
			}
		}
		
		return vals;
	}
	

	function getIdentityTransform() {
		var identityTransform = {};
		identityTransform.translate = [0, 0];
		identityTransform.scale = [1, 1];
		identityTransform.angle = 0;
		return identityTransform;
	}

	// get named puppetMeasurement
	function getPuppetMeasurement(self, inMeasurementName, args, viewIndex) {
		var measurement = null, puppetMeasurements = self.aPuppetMeasurements[viewIndex];
		
		if (inMeasurementName && puppetMeasurements[inMeasurementName]) {
			measurement = puppetMeasurements[inMeasurementName];

			// TODO: it's not ideal that some parameter adjustments happen here, while others
			// happen in computePuppetTransforms. should consolidate.
			// apply relevant parameter adjustments
			if (inMeasurementName === "noseDepth" || inMeasurementName === "eyeDepth") {
				measurement *= args.getParam("parallaxFactor") / 100;
			}
		}

		return measurement;
	} 

	// given a set of head14 params and a mapping entry with list of samples
	// determine the appropriate transform to return
	function computePuppetTransform(self, inHead14, inMappingEntry, args, viewIndex) {
		var finalTransform = getIdentityTransform(), transform, labels, translateUnit, samples, custom, customArgs, 
		translateRange, scaleRange, angleRange,
		i, j, v, inVec, vec1, vec2, range, offset, rangeNorm, alpha;

		for (i = 0; i < inMappingEntry.length; i += 1) {

			labels = inMappingEntry[i].labels;
			samples = inMappingEntry[i].samples;
			custom = inMappingEntry[i].custom;
			inVec = []; vec1 = []; vec2 = []; range = []; offset = [];
			translateRange = [0, 0];
			scaleRange = [0, 0];
			angleRange = 0;

			transform = getIdentityTransform();

			// interpolate based on mapping samples
			if (samples && samples.length >= 2) {

				V2.subtract(samples[samples.length-1].T.translate, samples[0].T.translate, translateRange);
				V2.subtract(samples[samples.length-1].T.scale, samples[0].T.scale, scaleRange);
				angleRange = samples[samples.length-1].T.angle - samples[0].T.angle;

				for (j = 0; j < labels.length; j += 1) {
					if (inHead14.hasOwnProperty(labels[j])) {
						inVec.push(inHead14[labels[j]]);
					}

					if (samples[0].head14.hasOwnProperty(labels[j])) {
						vec1.push(samples[0].head14[labels[j]]);
					}
					if (samples[samples.length-1].head14.hasOwnProperty(labels[j])) {
						vec2.push(samples[samples.length-1].head14[labels[j]]);
					}
				}

				if (inVec.length === labels.length && vec1.length === labels.length && vec2.length === labels.length) {

					rangeNorm = 0;
					for (j = 0; j < labels.length; j +=1 ) {
						range[j] = vec2[j] - vec1[j];
						offset[j] = inVec[j] - vec1[j];
						rangeNorm += (range[j] * range[j]);
					}

					rangeNorm = Math.sqrt(rangeNorm);

					// get dot product of offset onto range
					alpha = 0;
					for (j = 0; j < labels.length; j += 1) {
						alpha += offset[j] * (range[j] / rangeNorm);
					}
					alpha /= rangeNorm;

					// clamp
					if (alpha < 0) { alpha = 0; }
					if (alpha > 1) { alpha = 1; }

					// compute interpolated transform values
					V2.add(transform.translate, V2.add(samples[0].T.translate, V2.scale(alpha, translateRange, []), []), transform.translate);
					V2.xmy(transform.scale, V2.add(samples[0].T.scale, V2.scale(alpha, scaleRange, []), []), transform.scale);
					transform.angle += (samples[0].T.angle + (alpha * angleRange));
				}

			}
			else if (custom) {
				customArgs = [];
				for (j = 0; j < labels.length; j += 1) {
					v = inHead14[labels[j]];
					if (v === undefined) {
						console.log("missing FaceTracker custom value for " + labels[j]);
					} else {
						customArgs.push(v);
					}
				}

				custom(customArgs, transform);
			}

			// get translateUnit and scale transform
			translateUnit = getPuppetMeasurement(self, inMappingEntry[i].translateUnit, args, viewIndex);

			if (translateUnit) {
				V2.scale(translateUnit, transform.translate, transform.translate);
			}

			// add to finalTransform
			V2.add(finalTransform.translate, transform.translate, finalTransform.translate);
			V2.xmy(finalTransform.scale, transform.scale, finalTransform.scale);
			finalTransform.angle += transform.angle;
		}

		return finalTransform;
	}

	// get corresponding entry in head14ToPuppetTransformMapping
	function getHead14ToPuppetTransformEntry(inLabel) {
		var entry = null;
		if (head14ToPuppetTransformMapping.hasOwnProperty(inLabel)) {
			entry = head14ToPuppetTransformMapping[inLabel];
		}

		return entry;
	}

	// 1.0 means no change to the scale, 0.0 means scale will be identity (either 1.0 or -1.0), and 0.5
	//	means the scale will be half the strength (4 -> 2, 0.25 -> 0.5)
	function adjustScaleByFactor(scale, factor) {
		var negative, scaleDown;

		// handle 0.0 scale
		if (scale === 0) {
			if (factor > 0) {
				return scale;
			} else {
				return 1.0;
			}
		}

		if (factor === 1) {
			return scale;
		}
		
		negative = scale < 0;
		
		if (negative) {
			scale = -scale; // make it positive for a moment
		}
		
		scaleDown = scale < 1;
		
		if (scaleDown) {
			scale = 1/scale; // make it > 1 for a moment
		}
		
		// now we only need to deal with scale > 1 here
		scale = 1 + (scale - 1) * factor;

		if (scaleDown) {
			scale = 1/scale;
		}

		if (negative) {
			scale = -scale; // back to negative
		}
		
		return scale;
	}

	function adjustTransformByFactor(translate, factor) {
		return [translate[0] * factor, translate[1] * factor];
	}
	
	function applyFactorToTransform(inPosFactor, inScaleFactor, inRotFactor, inoutTransform) {
		// if getIdentityTransform() ever returns something other than actual identity, this function
		//	would need to be updated to do a LERP between that and the passed param
		inoutTransform.translate = adjustTransformByFactor(inoutTransform.translate, inPosFactor);
		inoutTransform.scale[0] = adjustScaleByFactor(inoutTransform.scale[0], inScaleFactor);
		inoutTransform.scale[1] = adjustScaleByFactor(inoutTransform.scale[1], inScaleFactor);
		inoutTransform.angle *= inRotFactor;
	}

	// customized version of applyParamFactorToNamedTransform that allows for adjusting pos, scale and rot factors.
	// the values in factors scale the effect of the param on the corresponding component of the transform.
	// e.g., factors.pos = 1 modifies translation by the param full param values, 
	//		and factors.pos = 0 does not modify translation at all
	function applyParamFactorToNamedTransformCustom(self, paramName, args, transforms, transformName, factors)
	{
		var t = transforms[transformName], factor;

		if (t) {
			factor = args.getParam(paramName) / 100;
			applyFactorToTransform(1 + (factor-1) * factors.pos, 1 + (factor-1) * factors.scale, 1 + (factor-1) * factors.rot, t);
		}		
	}

	// paramName is for a (currently root-level) param that has 100 as a "neutral" gain
	function applyParamFactorToNamedTransform(self, paramName, args, transforms, transformName) {
		var factors = { pos : 1, scale : 1, rot : 1 };
		applyParamFactorToNamedTransformCustom(self, paramName, args, transforms, transformName, factors);
	}

	function setTransformScale(transforms, transformName, scale) {
		var t = transforms[transformName];
		if (t) {
			t.scale = scale;
		}
	}

	/* unused
	function setTransformTranslate(transforms, transformName, translate) {
		var t = transforms[transformName];
		if (t) {
			t.translate = translate;
		}
	}*/

	function addTransformTranslate(transforms, transformName, translate) {
		var t = transforms[transformName];
		if (t) {
			t.translate[0] += translate[0];
			t.translate[1] += translate[1];
		}
	}

	// compute mouse eye gaze offset
	function computeNormalizedMouseEyeGazeOffset(self, args) {
		var mouseVec, mouseEyeGazeOffset = [0, 0], 
			deltaFromSceneCenter, distFromSceneCenter, angle,			
			epsilon = 0.00001,
			leftDownB = args.getParamEventValue("mouseEyeGaze", "Mouse/Down/Left"),
			mousePosition0 = args.getParamEventValue("mouseEyeGaze", "Mouse/Position");

		if (leftDownB && mousePosition0) {
			mouseVec = vec2(mousePosition0);
			args.setEventGraphParamRecordingValid("mouseEyeGaze");

			// eye gaze determined by mouse position wrt a circle with 200px radius
			// at the center of the scene. the perimeter of the circle represents 
			// the boundary of the eye.

			deltaFromSceneCenter = mouseVec;
			distFromSceneCenter = vec2.magnitude(deltaFromSceneCenter);
			deltaFromSceneCenter = vec2.normalize(deltaFromSceneCenter, vec2());

			// note: Math.atan seems to return a valid result (PI/2) even if you 
			// call it with NaN (e.g., 1/0), but just to be safe, we're checking
			// for deltaFromSceneCenter[0] > epsilon here and explicitly setting the angle. -wilmotli
			if (deltaFromSceneCenter[0] > epsilon) {
				angle = Math.atan(deltaFromSceneCenter[1]/deltaFromSceneCenter[0]); 
			} else if (deltaFromSceneCenter[0] < -epsilon) {
				angle = -Math.PI + Math.atan(deltaFromSceneCenter[1]/deltaFromSceneCenter[0]);
			} else {
				angle = (deltaFromSceneCenter[1] > 0) ? 0.5 * Math.PI : -0.5 * Math.PI;
			}

			// clamp to a 200px radius circle
			distFromSceneCenter /= 200;
			if (distFromSceneCenter > 1) distFromSceneCenter = 1;

			// compute offset vector
			mouseEyeGazeOffset[0] = distFromSceneCenter * Math.cos(angle);
			mouseEyeGazeOffset[1] = distFromSceneCenter * Math.sin(angle);
		}

		return mouseEyeGazeOffset;
	}

	// compute transforms for puppet
	function computePuppetTransforms(self, args, head14, viewIndex) {
		var transforms = {}, transform, headTransform,
			faceFeatureLabel, head14ToPuppetTransformEntry,
			featureLabels = self.aFeatureLabels[viewIndex], 
			normalizedMouseEyeGazeOffset, leftMouseEyeGazeOffset, rightMouseEyeGazeOffset, 
			leftEyeGazeRangeX = getPuppetMeasurement(self, "leftEyeGazeRangeX", args, viewIndex),
			leftEyeGazeRangeY = getPuppetMeasurement(self, "leftEyeGazeRangeY", args, viewIndex),
			rightEyeGazeRangeX = getPuppetMeasurement(self, "rightEyeGazeRangeX", args, viewIndex),
			rightEyeGazeRangeY = getPuppetMeasurement(self, "rightEyeGazeRangeY", args, viewIndex),
			mouseEyeGazeFactor = args.getParam("mouseEyeGazeFactor") / 100; 

		// apply parameter adjustments to puppet measurements
		for (faceFeatureLabel in featureLabels) {
			if (featureLabels.hasOwnProperty(faceFeatureLabel)) {
				head14ToPuppetTransformEntry = getHead14ToPuppetTransformEntry(faceFeatureLabel);
				if (featureLabels[faceFeatureLabel] && head14ToPuppetTransformEntry) {
					transform = computePuppetTransform(self, head14, head14ToPuppetTransformEntry, args, viewIndex);
					transforms[faceFeatureLabel] = transform;
				} 
			}
		}

		// 
		// adjust eyes
		//
		if (featureLabels["Adobe.Face.RightEyelidTop"] && featureLabels["Adobe.Face.RightEyelidBottom"]) {
			setTransformScale(transforms, "Adobe.Face.RightEye", [1, 1]);
		} else {
			// not clear why this clause is needed, though harmless
			transforms["Adobe.Face.RightEyelidTop"] = getIdentityTransform();
			transforms["Adobe.Face.RightEyelidBottom"] = getIdentityTransform();
		}
		if (featureLabels["Adobe.Face.LeftEyelidTop"] && featureLabels["Adobe.Face.LeftEyelidBottom"]) {
			setTransformScale(transforms, "Adobe.Face.LeftEye", [1, 1]);
		} else {
			transforms["Adobe.Face.LeftEyelidTop"] = getIdentityTransform();
			transforms["Adobe.Face.LeftEyelidBottom"] = getIdentityTransform();
		}
		
		applyParamFactorToNamedTransform(self, "eyebrowFactor", args, transforms, "Adobe.Face.LeftEyebrow");
		applyParamFactorToNamedTransform(self, "eyebrowFactor", args, transforms, "Adobe.Face.RightEyebrow");

		// when blink puppet is available, we don't want to scale the eye
		// TODO: this implies we should disable the eyeFactor param, but we currently don't have a 
		// way to do that (i.e. onCreateStageBehavior should return param UI hint that the param should be disabled)
		if (self.leftBlinkLayers[viewIndex].length > 0) {
			setTransformScale(transforms, "Adobe.Face.LeftEye", [1, 1]);
		} else {
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.LeftEye", { pos : 0, scale : 1, rot : 0 });
		}

		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.LeftEyelidTop", { pos : 1, scale : 0, rot : 0 });
		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.LeftEyelidBottom", { pos : 1, scale : 0, rot : 0 });			
		
		if (self.rightBlinkLayers[viewIndex].length > 0) {
			setTransformScale(transforms, "Adobe.Face.RightEye", [1, 1]);
		} else {
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.RightEye", { pos : 0, scale : 1, rot : 0 });
		}

		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.RightEyelidTop", { pos : 1, scale : 0, rot : 0 });
		applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Adobe.Face.RightEyelidBottom", { pos : 1, scale : 0, rot : 0 });			

		//
		// adjust eye gaze
		//
		applyParamFactorToNamedTransformCustom(self, "eyeGazeFactor",	args, transforms, "Adobe.Face.LeftPupil", { pos : 1, scale : 0, rot : 0 });
		applyParamFactorToNamedTransformCustom(self, "eyeGazeFactor",	args, transforms, "Adobe.Face.RightPupil", { pos : 1, scale : 0, rot : 0 });

		// add mouse-based eye gaze
		normalizedMouseEyeGazeOffset = computeNormalizedMouseEyeGazeOffset(self, args);
		leftMouseEyeGazeOffset = vec2.initWithEntries(normalizedMouseEyeGazeOffset[0] * leftEyeGazeRangeX * mouseEyeGazeFactor, normalizedMouseEyeGazeOffset[1] * leftEyeGazeRangeY * mouseEyeGazeFactor, vec2());
		rightMouseEyeGazeOffset = vec2.initWithEntries(normalizedMouseEyeGazeOffset[0] * rightEyeGazeRangeX * mouseEyeGazeFactor, normalizedMouseEyeGazeOffset[1] * rightEyeGazeRangeY * mouseEyeGazeFactor, vec2());
		//
		addTransformTranslate(transforms, "Adobe.Face.RightPupil", leftMouseEyeGazeOffset);
		addTransformTranslate(transforms, "Adobe.Face.LeftPupil", rightMouseEyeGazeOffset);

		//
		// adjust mouth
		//
		applyParamFactorToNamedTransform(self, "mouthFactor", args, transforms, "Adobe.Face.Mouth");

		//
		// adjust head
		//
		setTransformScale(transforms, "Adobe.Face.Head", [head14["Head/Scale"], head14["Head/Scale"]]);

		headTransform = transforms["Adobe.Face.Head"];
		if (headTransform) {
			applyFactorToTransform(
				args.getParam("headPosFactor") / 100, 
				args.getParam("headScaleFactor") / 100, 
				args.getParam("headRotFactor") / 100, 
				headTransform);
		}

		// DEBUG
		/* (uncomment var everyN far above)
		everyN += 1;
		if (everyN >= 1) {
			everyN = 0;
			// Put debugging output here
		}
		*/

		return transforms;
	}


	function applyPuppetTransforms(self, args, viewIndex, inTransforms) {
		var faceFeatureLabel, transform, initState, faceFeatureNode, 
			position = [], scale = [], angle;

		for (faceFeatureLabel in inTransforms) {
			if (inTransforms.hasOwnProperty(faceFeatureLabel)) {
				transform = inTransforms[faceFeatureLabel];
				initState = self.aPuppetInitTransforms[viewIndex][faceFeatureLabel];

				faceFeatureNode = getHandle(args, faceFeatureLabel, viewIndex);

				if (transform && initState && faceFeatureNode) {
					V2.add(initState.position, V2.scale(1, transform.translate), position);
					// TODO: will initState ever have a scale != 1 that we have to take into account?
					scale = transform.scale;
					angle = initState.angle + transform.angle;

					var transformTask = {
						x : position[0],
						y : position[1],
						xScale : scale[0],
						yScale : scale[1],
						angle : angle
					};

					// FIXME: connect with blend weight...
					var	weight = 1.0,
						move = new tasks.MoveTo(transformTask);
					tasks.handle.attachTask(faceFeatureNode, move, weight);

				}
			}
		}
	}	

	
	function makeValidIdFromLabel (str) {
		return str.replace(/[^\w]/g, "_");
	}
	
	function makeHandleIdFromLabel (str) {
		return "H_" + makeValidIdFromLabel(str);
	}
	
	function makeLayerIdFromLabel (str) {
		return "L_" + makeValidIdFromLabel(str);
	}
	
	function addHiddenLayerParam (aParams, aMatchIn, label, tooltip) {
		var firstMatch = aMatchIn[0];
		var aMatches = [];
		aMatchIn.forEach(function (match) {
			aMatches.push("//" + match);
		});
		
		aParams.push({id:makeLayerIdFromLabel(firstMatch), type:"layer", uiName:label,
					dephault:{match:aMatches, startMatchingAtParam: "viewLayers"},
					maxCount:1, uiToolTip:tooltip, hidden:true});
	}
	
	function addLayerParamAtIndex (aParams, aMatchIn, label, tooltip, index) {
		var firstMatch = aMatchIn[0];
		var aMatches = [];
		var param = {id:makeLayerIdFromLabel(firstMatch), type:"layer", uiName:label,
					dephault:{match:aMatches, startMatchingAtParam: "viewLayers"},
					maxCount:1, uiToolTip:tooltip, hidden:false};

		aMatchIn.forEach(function (match) {
			aMatches.push("//" + match);
		});
		
		aParams.splice(index, 0, param);
	}
	
	function defineHandleParams () {
		var aParams = [];
		var rightIndex = -1, leftIndex = -1;
		
		faceFeatureTagDefinitions.forEach( function (tagDefn) {
			var label = tagDefn.id;
			var def = {id:makeHandleIdFromLabel(label), type:"handle", uiName:tagDefn.uiName,
						dephault:{match:"//"+label},
						hidden:false};
			
			if (label === "Adobe.Face.LeftPupil") {
				leftIndex = aParams.length+1;
			} else if (label === "Adobe.Face.RightPupil") {
				rightIndex = aParams.length+1;
			}
			
			if (label !== "Adobe.Face.Head") {	// special case for head, as it is usually a parent of the views,
			 	def.maxCount = 1;				//	so we don't depend on it being inside a view (but do allow it)
				def.dephault.startMatchingAtParam = "viewLayers";
			}
			
			aParams.push(def);
		});

			
		addLayerParamAtIndex(aParams, ["Adobe.Face.RightPupilRange"], 
			"$$$/animal/behavior/face/tag/RightPupilRange=Right Pupil Range", 
			"$$$/animal/behavior/face/tag/RightPupilRange/tooltip=Sets the movement range of the Right Pupil", rightIndex);

		addLayerParamAtIndex(aParams, ["Adobe.Face.LeftPupilRange"],
			"$$$/animal/behavior/face/tag/LeftPupilRange=Left Pupil Range",
			"$$$/animal/behavior/face/tag/LeftPupilRange/tooltip=Sets the movement range of the Left Pupil", leftIndex+1);	// adding 1 here because previous add moved these indices down

		// and layer params for the pupils
		addHiddenLayerParam(aParams, ["Adobe.Face.LeftPupilSize"], 
			"Left Pupil", 
			"Used to compute range of the Left Pupil");
		addHiddenLayerParam(aParams, ["Adobe.Face.RightPupilSize"], 
			"Right Pupil", 
			"Used to compute range of the Right Pupil");

		// and layer params for the eyelids (TODO: consolidate to only have layer params, not handles?)
		
		addHiddenLayerParam(aParams, ["Adobe.Face.LeftEyelidTopLayer"], 
			"Left Eyelid Top", 
			"Sets the vertical range of the Left Eyelid; if missing, Left Eyelid Top _handle_ is used instead");
		
		addHiddenLayerParam(aParams, ["Adobe.Face.LeftEyelidBottomLayer"], 
			"Left Eyelid Bottom", 
			"Sets the vertical range of the Left Eyelid; if missing, Left Eyelid Bottom _handle_ is used instead");
		
		addHiddenLayerParam(aParams, ["Adobe.Face.RightEyelidTopLayer"], 
			"Right Eyelid Top", 
			"Sets the vertical range of the Right Eyelid; if missing, Right Eyelid Top _handle_ is used instead");
		
		addHiddenLayerParam(aParams, ["Adobe.Face.RightEyelidBottomLayer"], 
			"Right Eyelid Bottom", 
			"Sets the vertical range of the Right Eyelid; if missing, Right Eyelid Bottom _handle_ is used instead");
		
		return aParams;
	}


	function defineMouthLayerParams () {
		var aParams = [];
		
		mouthShapeLayerTagDefinitions.forEach( function (tagDefn) {
			aParams.push({id:makeLayerIdFromLabel(tagDefn.id), type:"layer", uiName:tagDefn.uiName,
						dephault:{match:"//"+tagDefn.id,  startMatchingAtParam:"mouthsParent"},
						maxCount:1}); // only one of each mouth shape per "Mouth" group
		});
		
		var mouthParentTagDefn = mouthParentLayerTagDefinition[0];
		// add this to the front, after sorting the rest
		aParams.splice(0, 0, {id:"mouthsParent", // TODO: use startMatchingAtParam: "viewLayers" but first need support for more than one sMAP
							type:"layer",
							uiName:mouthParentTagDefn.uiName,
							dephault:{match:"//"+mouthParentTagDefn.id}});		
		
		return aParams;
	}


	// TODO: factor with LipSync.js
	// returns array of maps from mouth label to matched layer, one array per mouth group
	// called during onCreate, so it uses getStaticParam
	//	outer array is one per mouth group (i.e. number of matches of mouthsParent)
	function getMouthsCandidates (args) {
		var aMouthsLayers = [],
			bHideSiblings = true,
			aMouths = args.getStaticParam("mouthsParent");
		
		aMouths.forEach(function (mouthLayer, mouthGroupIndex) {
			var mouths = { parent: mouthLayer };	// map from label -> layer
			mouthShapeLabels.forEach(function (label) {
				var aaLayers = args.getStaticParam(makeLayerIdFromLabel(label)),
					layer = aaLayers[mouthGroupIndex][0]; // we only support one of each shape per mouth
				
				if (layer) {
					mouths[label] = layer;
					layer.setTriggerable(bHideSiblings);
				}
			});
			aMouthsLayers.push(mouths);
		});
		
		return aMouthsLayers;
	}

	
	// returns actual mouth node for requested mouth index, which may not exist (uses fallbacks in mouthShapeCascades)
	function getClosestValidMouthContainerRoot(args, groupIndex, inMouthIndex) {

		var mouthIndicesToTry = [], i, mouthIndex, mouthShapeLabel,
			aLayers, layer;

		if (mouthShapeCascades.hasOwnProperty(inMouthIndex)) {
			mouthIndicesToTry = mouthShapeCascades[inMouthIndex];
		}
		else {
			mouthIndicesToTry = [inMouthIndex, 0];
		}

		for (i = 0; i < mouthIndicesToTry.length; i += 1) {
			mouthIndex = mouthIndicesToTry[i];
			if (mouthIndex >= 0 && mouthIndex < mouthShapeLabels.length) {
				mouthShapeLabel = mouthShapeLabels[mouthIndex];
				if (!mouthShapeLabel) {
					// this shouldn't happen, so adding error message if it does to help diagnose
					console.log("invalid mouth index: " + mouthIndex);
				} else {
					aLayers = args.getParam(makeLayerIdFromLabel(mouthShapeLabel))[groupIndex];
					layer = aLayers[0];

					if (layer) {
						return layer;
					}
				}
			}
		}

		return null;
	}

	
	function chooseMouthReplacements (args, aMouthsGroups, head14) {
		var priority = 0.25,
			mouthIndexToShow = Math.max(head14["Head/MouthShape"], 0);
		
		if (mouthIndexToShow >= 0) {
			aMouthsGroups.forEach(function (mouthsGroup, groupIndex) {
				// get valid (found) mouth that is closest to specified mouth to show
				var validMouthLayer = getClosestValidMouthContainerRoot(args, groupIndex, mouthIndexToShow);

				if (validMouthLayer) {
					validMouthLayer.trigger(priority); // compare priority to LipSync & KeyReplacer
				} else {
					// alternatively, we could output error msg saying that we can't find a Neutral mouth
					//console.logToUser("choseMouthReplacements(): no Neutral mouth found");
				}
			});
		}
	}

	function chooseBlinkReplacements(aaLayers, eyeOpenness, threshold, eyeFactor) {
		aaLayers.forEach(function (aLays) {
			aLays.forEach(function (blinkLayer) {
				if ((eyeOpenness <= threshold) && (eyeFactor !== 0))  { // the eyelid measurment in head14 never seems to go below about .3
					blinkLayer.trigger();
				}
			});
		});
	}

	function getHandle (args, label, viewIndex) {
		var v = args.getParam(makeHandleIdFromLabel(label));
		
		if (label === "Adobe.Face.Head") {
			// special case: no startMatchingAtParam, but viewIndex is not the right way to span
			//	this array -- it's of unrelated length. Still, in practice,
			//	honoring it here means Head handles will actually move as long there are the
			//	same number or more views.
			//	TODO: find real fix for this, either by pulling access to this handle into
			//		a separate loop called once for each match, or figure out how to have Head
			//		not be a special view-less case (and still support the common practice of
			//		having multiple views inside a single Head group).
			return v[viewIndex];	// might be undefined
		} else {
			// only works for a single handle (per view) for each label right now
			utils.assert(Array.isArray(v));
			return v[viewIndex][0];	// might be undefined
		}
	}

	function computeInitTransforms (self, args, view, viewIndex) {
		var faceFeatureLabel, handleParam,
			featureLabels = self.aFeatureLabels[viewIndex],
			puppetInitTransforms = {};

		for (faceFeatureLabel in featureLabels) {
			if (featureLabels.hasOwnProperty(faceFeatureLabel)) {
				handleParam = getHandle(args, faceFeatureLabel, viewIndex);
				if (handleParam) {
					// WL: we are explicitly recording whether a face feature label 
					// is present ... as an alternative, we could also just check 
					// if puppetInitTransforms has an entry for the label.
					featureLabels[faceFeatureLabel] = true;

					puppetInitTransforms[faceFeatureLabel] = {
						"position": [0, 0],
						"scale"	: [1, 1],
						"angle"	: 0
					};
				}
			}
		}
		
		utils.assert(self.aPuppetInitTransforms.length === viewIndex); // assumes we're called in view order
		self.aPuppetInitTransforms.push(puppetInitTransforms);
	}
	
	// returns the first match from the layer param, or null
	function getLayer (args, label, viewIndex) {
		var match = args.getParam(makeLayerIdFromLabel(label))[viewIndex];
		
		if (match) {
			match = match[0];	// only get first element from array if there is one (might be an empty view)
		}
		
		if (!match) {
			match = null;					// the old code returned null
		}
		
		return match;
	}

	function computeEyeGazeRanges (self, args, viewIndex) {
		var puppetMeasurements = self.aPuppetMeasurements[viewIndex], 
			leftEyeGazeRangeX = 0, leftEyeGazeRangeY = 0, rightEyeGazeRangeX = 0, rightEyeGazeRangeY = 0,
			leftEyeWidth = getPuppetMeasurement(self, "leftEyeWidth", args, viewIndex),
			leftEyeHeight = getPuppetMeasurement(self, "leftEyeHeight", args, viewIndex),
			rightEyeWidth = getPuppetMeasurement(self, "rightEyeWidth", args, viewIndex),
			rightEyeHeight = getPuppetMeasurement(self, "rightEyeHeight", args, viewIndex),
			leftPupilWidth = getPuppetMeasurement(self, "leftPupilWidth", args, viewIndex),
			leftPupilHeight = getPuppetMeasurement(self, "leftPupilHeight", args, viewIndex),
			rightPupilWidth = getPuppetMeasurement(self, "rightPupilWidth", args, viewIndex),
			rightPupilHeight = getPuppetMeasurement(self, "rightPupilHeight", args, viewIndex);

		if (leftEyeWidth) {
			leftEyeGazeRangeX = (leftPupilWidth) ? 0.5 * (leftEyeWidth - leftPupilWidth) : 0.5 * leftEyeWidth;
		}
		if (leftEyeHeight) {
			leftEyeGazeRangeY = (leftPupilHeight) ? 0.5 * (leftEyeHeight - leftPupilHeight) : 0.5 * leftEyeHeight;
		}
		if (rightEyeWidth) {
			rightEyeGazeRangeX = (rightPupilWidth) ? 0.5 * (rightEyeWidth - rightPupilWidth) : 0.5 * rightEyeWidth;
		}
		if (rightEyeHeight) {
			rightEyeGazeRangeY = (rightPupilHeight) ? 0.5 * (rightEyeHeight - rightPupilHeight) : 0.5 * rightEyeHeight;
		}

		puppetMeasurements["leftEyeGazeRangeX"] = leftEyeGazeRangeX;
		puppetMeasurements["leftEyeGazeRangeY"] = leftEyeGazeRangeY;
		puppetMeasurements["rightEyeGazeRangeX"] = rightEyeGazeRangeX;
		puppetMeasurements["rightEyeGazeRangeY"] = rightEyeGazeRangeY;
	}
	
	function computePuppetMeasurements (self, args, viewIndex) {
		function getLayerParam(id) {
			return getLayer(args, id, viewIndex);
		}
		function getHandleParam(id) {
			return getHandle(args, id, viewIndex);
		}
		
		function getHandleParentLayerFallback (id) {
			var layer = null, handle = getHandleParam(id);
			if (handle) {
				layer = handle.getPuppet().getParentLayer().getSdkLayer();
			}
			return layer;
		}

		// "pos" measures the x or y dist between two rest positions
		// "bbox" measures the x or y dimension of a rest bbox
		// "scaledCopy" copies another named measurement scaled by factor
		
		var measurementsList = [
				["headWidth", "bbox", self.headPuppet, "X"],
				["headHeight", "bbox", self.headPuppet, "Y"],
			
				["leftEyeEyebrowDist",	"pos", getHandleParam("Adobe.Face.LeftEye"),	getHandleParam("Adobe.Face.LeftEyebrow"),		"Y"],
				["leftEyeEyebrowDist",	"scaledCopy", 0.15, "headHeight"], // fallback if both "Left Eye" and "Left Eyebrow" handles don't exist
																			// maybe remove this -- is it useful to have eyebrows without eyes?

				["rightEyeEyebrowDist", "pos", getHandleParam("Adobe.Face.RightEye"),		getHandleParam("Adobe.Face.RightEyebrow"),	"Y"],
				["rightEyeEyebrowDist", "scaledCopy", 0.15, "headHeight"], // fallback if both "Right Eye" and "Right Eyebrow" handles don't exist
			
				["interocularDist",		"pos", getHandleParam("Adobe.Face.LeftEye"),		getHandleParam("Adobe.Face.RightEye"),		"X"],
				["interocularDist",		"scaledCopy", 0.25, "headWidth"],
			
				["noseDepth",		"scaledCopy", 0.15, "interocularDist"],
				["eyeDepth",		"scaledCopy", 0.09, "interocularDist"],
			
				// controls distance that the left eyelid top moves down, and bottom moves up; primary: space between top & bottom bounds
				["leftEyelidDist",	"bboxDiff", getLayerParam("Adobe.Face.LeftEyelidTopLayer"), getLayerParam("Adobe.Face.LeftEyelidBottomLayer"), "Y"],
				//	fallback: distance between top & bottom handles
				["leftEyelidDist",	"pos", getHandleParam("Adobe.Face.LeftEyelidTop"), getHandleParam("Adobe.Face.LeftEyelidBottom"), "Y"],
			
				["rightEyelidDist", "bboxDiff", getLayerParam("Adobe.Face.RightEyelidTopLayer"), getLayerParam("Adobe.Face.RightEyelidBottomLayer"), "Y"],
				["rightEyelidDist", "pos", getHandleParam("Adobe.Face.RightEyelidTop"), getHandleParam("Adobe.Face.RightEyelidBottom"), "Y"],
			
				["leftEyeWidth",	"bbox", getLayerParam("Adobe.Face.LeftPupilRange"), "X"],
				["leftEyeWidth",	"bbox", getHandleParentLayerFallback("Adobe.Face.LeftEye"), "X"],
			
				["rightEyeWidth",	"bbox", getLayerParam("Adobe.Face.RightPupilRange"), "X"],
				["rightEyeWidth",	"bbox", getHandleParentLayerFallback("Adobe.Face.RightEye"), "X"],
			
				["leftEyeHeight",	"bbox", getLayerParam("Adobe.Face.LeftPupilRange"), "Y"],
				["leftEyeHeight",	"bbox", getHandleParentLayerFallback("Adobe.Face.LeftEye"), "Y"],
			
				["rightEyeHeight",	"bbox", getLayerParam("Adobe.Face.RightPupilRange"), "Y"],
				["rightEyeHeight",	"bbox", getHandleParentLayerFallback("Adobe.Face.RightEye"), "Y"],
			
				["leftPupilWidth",	"bbox", getLayerParam("Adobe.Face.LeftPupilSize"), "X"],
				["leftPupilHeight", "bbox", getLayerParam("Adobe.Face.LeftPupilSize"), "Y"],

				["rightPupilWidth", "bbox", getLayerParam("Adobe.Face.RightPupilSize"), "X"],
				["rightPupilHeight", "bbox", getLayerParam("Adobe.Face.RightPupilSize"), "Y"],				

				// $$$ TODO: the 0 below implies there is always a Neutral mouth in the first Mouth group
				// note: the sMAP parent for mouth is not view, only Mouth group
				["mouthWidth", "bbox", getLayer(args, "Adobe.Face.NeutralMouth", 0), "X"], // TODO: change to mouth group?			
				["mouthHeight", "bbox", getLayer(args, "Adobe.Face.NeutralMouth", 0), "Y"] 
			],

			puppetMeasurements = {},
			i, j, value, m, bbox1, bbox2;

		for (i = 0; i < measurementsList.length; i += 1) {
			m = measurementsList[i];

			if (!(puppetMeasurements.hasOwnProperty(m[0]) && puppetMeasurements[m[0]])) {
				value = null;

				if (m[1] === "pos" && m[2] && m[3] && m[4]) {
					// rootMatrix is a 3x3 transformation (in homogeneous coords)
					// rootMatrix[6] = x translation, rootMatrix[7] = y translation 
					if (m[4] === "X") { 
						j = 6; 
					}
					else { 
						j = 7; 
					}
					// TODO: may not work as intended when Puppets are scaled: ten times smaller or ten times larger
					value = Math.abs(args.getHandleMatrixRelativeToScene(m[2])[j]-args.getHandleMatrixRelativeToScene(m[3])[j]);
				} 
				else if (m[1] === "bbox" && m[2] && m[3]) {
					bbox1 = m[2].getBounds();

					if (m[3] === "X") { 
						value = bbox1[2]; 
					}
					else { 
						value = bbox1[3]; 
					}
				}
				else if (m[1] === "bboxDiff" && m[2] && m[3] && m[4]) {

					bbox1 = m[2].getBounds();
					bbox2 = m[3].getBounds();

					if (m[4] === "X") {
						value = bbox2[0] - (bbox1[0] + bbox1[2]);
					}
					else {
						value = bbox2[1] - (bbox1[1] + bbox1[3]);
					}
					if (value <= 0) {
						value = null;	// will fall back on "pos" instead
					}
				}
				else if (m[1] === "scaledCopy" && m[2] && m[3]) {
					value = m[2] * puppetMeasurements[m[3]];
				}

				puppetMeasurements[m[0]] = value;

				// DEBUG
				//console.logToUser(m[0] + "_" + m[1] + " = " + puppetMeasurements[m[0]]);
			}
		}
		
		utils.assert(self.aPuppetMeasurements.length === viewIndex);
		self.aPuppetMeasurements.push(puppetMeasurements);

		// add additional compound measurements; note: reads aPuppetMeasurements
		computeEyeGazeRanges(self, args, viewIndex);
	}
	
	function setAllLayersAsTriggerableHideSibs(aaLays) {
		var bHideSiblings = true;

		aaLays.forEach(function (aLays) {
			aLays.forEach(function (lay) {
				lay.setTriggerable(bHideSiblings);
			});
		});
	}

	function setTriggerableLayers(self, args) {
		
		setAllLayersAsTriggerableHideSibs(self.leftBlinkLayers);

		setAllLayersAsTriggerableHideSibs(self.rightBlinkLayers);

		// gather replacement mouth layers, also sets as triggerable
		self.aMouthsGroups = getMouthsCandidates(args); // array of maps from label -> layer, and .parent too
	}

	/*function printPuppetContainerHandleHierarchies(inPuppet) {

		inPuppet.breadthFirstEach(function (p) {
			var cTree, hTree;
			console.log("p = " + p.getName());
			cTree = p.getContainerTree();

			if (cTree) {
				cTree.breadthFirstEach(function (c) {
					console.log("\tc = " + c.getName());
				});
			}

			hTree = p.getHandleTreeRoot();
			if (hTree) {
				console.log("\thTree:");				
				hTree.breadthFirstEach(function (h) {
					console.log("\t\th = " + h.getName());
				});
			}
		});
	}*/

	function chooseFaceTrackerReplacements(self, args, head14) {
		var eyeFactor = args.getParam("eyeFactor"),
			eyeClosedThresh = 0.45;
		
		chooseMouthReplacements(args, self.aMouthsGroups, head14);
		
		var leftEyelid = head14["Head/LeftEyelid"],
			rightEyelid = head14["Head/RightEyelid"];
		
		if (args.getParam("blinkOnly")) {
			leftEyelid = rightEyelid = (leftEyelid + rightEyelid) / 2;
		}

		chooseBlinkReplacements(self.leftBlinkLayers, leftEyelid, eyeClosedThresh, eyeFactor);
		chooseBlinkReplacements(self.rightBlinkLayers, rightEyelid, eyeClosedThresh, eyeFactor);
	}

	function animateWithFaceTracker(self, args) {
		var head14 = getFilteredHead14(args);

		// if we end up putting mouths under views, this will need to go into the loop just below
		chooseFaceTrackerReplacements(self, args, head14);
		
		self.aViews.forEach(function (view, viewIndex) {		
			var transforms = computePuppetTransforms(self, args, head14, viewIndex);
			
			applyPuppetTransforms(self, args, viewIndex, transforms);
		});
	}
	
	function canIterpolateHeadLabel (inHeadLabel) {
		return (inHeadLabel !== "Head/MouthShape");
	}	

	function FaceSmoothingFilter () {
		this.ringBufferSize = 3;
		this.ringBufferA = [null, null, null];	// To reduce latency, we always use the most recent values as our 4 variable.
		this.insertionIndex = 0;
		this.lastInsertionTime = null;
	}

	utils.mixin(FaceSmoothingFilter, {
		getInsertionIndex : function () {
			return this.insertionIndex;
		},
		getLastInsertionTime : function () {
			return this.lastInsertionTime;
		},
		setLastInsertionTime : function (t) {
			this.lastInsertionTime = t;
		},
		incrementInsertionIndex : function () {
			this.insertionIndex = (this.getInsertionIndex()+1) % this.ringBufferSize;
		},
		addCurrentSample : function (currentHead14, sampleTime, sampleInterval) {
			var insertKeyframeB = false, keyFramesA = [], 
					dt, interpTime;

			insertKeyframeB = (this.getLastInsertionTime() === null);
			if (!insertKeyframeB) {
				dt = sampleTime - this.getLastInsertionTime();

				insertKeyframeB = (dt >= sampleInterval);
			}
			if (insertKeyframeB) {
				dt = 0;
				this.ringBufferA[this.getInsertionIndex()] = currentHead14;
				this.incrementInsertionIndex();
				this.setLastInsertionTime(sampleTime);
			}

			keyFramesA[0] = currentHead14;
			for (var i = 1; i < this.ringBufferSize+1; i++) {
				var idx = (this.getInsertionIndex() - i + this.ringBufferSize);
				idx = idx % this.ringBufferSize;
				keyFramesA[i] = this.ringBufferA[idx];

				if (keyFramesA[i] === null) {
					utils.assert(i > 0);
					keyFramesA[i] = keyFramesA[i-1];
				}
			}

			interpTime = 1 - (dt / sampleInterval);
			interpTime = Math.max(0.0, interpTime);
			interpTime = Math.min(1.0, interpTime);

			var head14 = [];
			for (var headLabel in head14Labels) {
				if (head14Labels.hasOwnProperty(headLabel)) {
					
					if (canIterpolateHeadLabel(headLabel)) {
						var x0 = keyFramesA[0][headLabel], x1 = keyFramesA[1][headLabel], 
							x2 = keyFramesA[2][headLabel], x3 = keyFramesA[3][headLabel];

						head14[headLabel] = mathUtils.cubicLerp(x0, x1, x2, x3, interpTime, -0.5);
					} else {
						head14[headLabel] = keyFramesA[0][headLabel];
					}
				}
			}
			return head14;
		}
	});		
	
	return {
		about:			"$$$/private/animal/Behavior/FaceTracker/About=Face Tracker, (c) 2015.",
		description:	"$$$/animal/Behavior/FaceTracker/Desc=Controls head, eyes, eyebrows, nose, and mouth via your webcam",
		uiName:			"$$$/animal/Behavior/FaceTrackerOld/UIName=Face (obsolete)",
		defaultArmedForRecordOn: true,
		hideInBehaviorList: true,
	
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
		  return [
			{ id: "cameraInput", type: "eventGraph", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/cameraInput=Camera Input",
			 	inputKeysArray: ["Head/", pKeyCode],
		        outputKeyTraits: {
	                takeGroupsArray: [
		                {
		                    id: "Filtered/Head/*"
		                }
	                ]
		        },			
				uiToolTip: "$$$/animal/Behavior/FaceTracker/param/cameraInput/tooltip=Analyzed face data from the camera; pause the face pose by holding down semicolon (;)",
				defaultArmedForRecordOn: true, supportsBlending: true},
			{ id: "mouseEyeGaze", type: "eventGraph", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/mouseEyeGaze=Eye Gaze via Mouse Input", inputKeysArray: ["Mouse/"],
				uiToolTip: "$$$/animal/behavior/face/param/mouseEyeGaze/tooltip=Control eye gaze direction with your mouse" },
			{ id: "smoothingRate", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/smoothingRate=Smoothing", uiUnits: "%", min: 0, max: 100, precision: 0, dephault: 10, hideRecordButton: true,
				uiToolTip: "$$$/animal/behavior/face/param/smoothingRate/tooltip=Exaggerate or minimize the smoothness when transitioning between different face or head poses" },
			{ id: "headPosFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/headPosFactor=Head Position Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/headPosFactor/tooltip=Exaggerate or minimize how much the head moves when you move your head" },
			{ id: "headScaleFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/headScaleFactor=Head Scale Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/headScaleFactor/tooltip=Exaggerate or minimize how much the head scales when you move your head closer or farther from the camera" },
			{ id: "headRotFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/headRotFactor=Head Tilt Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/headRotFactor/tooltip=Exaggerate or minimize how much the head rotates when you tilt the top of your head to the right and left" },
			{ id: "eyebrowFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/eyebrowFactor=Eyebrow Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/eyebrowFactor/tooltip=Exaggerate or minimize how much the eyebrows move when you raise and lower your eyebrows" },
			{ id: "eyeFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/eyeFactor=Eyelid Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/eyeFactor/tooltip=Exaggerate or minimize how far the eyelids move (if present) or the eyes scale (if not) when you blink" },
			{ id: "eyeGazeFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/eyeGazeFactor=Eye Gaze Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/eyeGazeFactor/tooltip=Exaggerate or minimize how far the pupils move when you look around" },
			{ id: "mouseEyeGazeFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/mouseEyeGazeFactor=Eye Gaze Mouse Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/mouseEyeGazeFactor/tooltip=Exaggerate or minimize how far the pupils move when you move mouse around" },
			{ id: "mouthFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/mouthFactor=Mouth Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 0,
				uiToolTip: "$$$/animal/behavior/face/param/mouthFactor/tooltip=Exaggerate or minimize how much the mouth scales and moves to correspond with your mouth movements" },
			{ id: "parallaxFactor", type: "slider", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/parallaxFactor=Parallax Strength", uiUnits: "%", min: 0, max: 999, precision: 0, dephault: 100,
				uiToolTip: "$$$/animal/behavior/face/param/parallaxFactor/tooltip=Exaggerate or minimize the eyes and nose movement when you rotate your head left and right" },
			{ id: "viewLayers", type: "layer", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/Views=Views", dephault: { match:["//Adobe.Face.LeftProfile|Adobe.Face.LeftQuarter|Adobe.Face.Front|Adobe.Face.RightQuarter|Adobe.Face.RightProfile|Adobe.Face.Upward|Adobe.Face.Downward", "."] } },
			{ id: "handlesGroup", type: "group", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/handlesGroup=Handles", groupChildren: defineHandleParams() },
			{ id: "replacementsGroup", type: "group", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/replacementsGroup=Replacements", groupChildren: [
				{ id: "leftBlinkLayers", type: "layer", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/leftBlinkLayers=Left Blink", dephault: { match: "//Adobe.Face.LeftBlink", startMatchingAtParam: "viewLayers" }, maxCount: 1 },
				{ id: "rightBlinkLayers", type: "layer", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/rightBlinkLayers=Right Blink", dephault: { match: "//Adobe.Face.RightBlink", startMatchingAtParam: "viewLayers" }, maxCount: 1 },
				{ id: "blinkOnly", type: "checkbox", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/blinkOnly=Blink Eyes Together", dephault: true },
				{ id: "mouthGroup", type: "group", uiName: "$$$/animal/Behavior/FaceTracker/Parameter/mouthGroup=Mouths", groupChildren: defineMouthLayerParams() }
			]}
		  ];
		},
		
		defineTags: function () {
			var aAllTags = [];
			var aAllTagRefs = [faceFeatureTagDefinitions, faceLayerTagDefinitions, mouthShapeLayerTagDefinitions, mouthParentLayerTagDefinition, viewLayerTagDefinitions, miscLayerTagDefinitions];
			aAllTagRefs.forEach(function (ar) {
				aAllTags = aAllTags.concat(ar);
			}); 
			return {aTags:aAllTags};
		},
		
		onCreateBackStageBehavior: function (/*self*/) {
			return { order: 0.51, importance : 0.0 }; // must come after LipSync
		},
		
		onCreateStageBehavior: function (self, args) {
			var headHandle, newHead = null;
			args.getParam = args.getStaticParam;	// @@@HACK! until we rename this function
			
			this.onResetRehearsalData(self);

			// these arrays are all indexed by view
			self.aPuppetInitTransforms = [];
			self.aPuppetMeasurements = [];
			self.aFeatureLabels = [];

			headHandle = args.getParam(makeHandleIdFromLabel("Adobe.Face.Head"))[0];
			if (headHandle) {
				newHead = headHandle.getPuppet(); // get puppet that has the first Head handle
				//	avoids the need for an extra Head layer param, but also favors the first one
				//	hopefully OK as it's only used a fallback for when eye handles can't be found
			}
			
			self.headPuppet = newHead;	// used for fallback eyebrow measurement, might be null

			// find blink replacements
			self.leftBlinkLayers = args.getStaticParam("leftBlinkLayers");
			self.rightBlinkLayers = args.getStaticParam("rightBlinkLayers");
			
			self.aViews = args.getStaticParam("viewLayers");
			
			self.aViews.forEach(function (view, viewIndex) {
				var featureMap = {};
				faceFeatureTagDefinitions.forEach( function (tagDefn) { 
					featureMap[tagDefn.id] = false;
				});
			
				self.aFeatureLabels.push(featureMap);
				computeInitTransforms(self, args, view, viewIndex);
				computePuppetMeasurements(self, args, viewIndex);
			});
			
			setTriggerableLayers(self, args); // not view-multiplexed right now; only mouth-multiplexed

			// debug -- show puppet/container hierarchy
			// printPuppetContainerHandleHierarchies(stagePuppet);
		},
		// Clear the rehearsal state
		onResetRehearsalData : function (self) {
			self.currentTimeFaceSmoothingFilter = new FaceSmoothingFilter();
			self.currentPose0 = null;
			self.previousPose0 = null;
			self.lastValue0 = null;
			self.lastPauseKeyState = 0;
			
			// Used to store data related to transitioning between current value and pose, then pose back to current. 
			self.transitionStartTime = 0;
			self.transitionEndTime = 0;
		},

		onFilterLiveInputs: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			var inputParamId = "cameraInput", inputLiveB = args.isParamEventLive(inputParamId), paramOutputKey = args.getParamEventOutputKey(inputParamId);
			
			if (inputLiveB) {
				var smoothingRate = args.getParam("smoothingRate"), smoothingRateHz, head14, currentHead14 = getHead14(args),
					pauseKeyDown = (args.getParamEventValue(inputParamId, pKeyCode, null, null, true) || 0) && 1, currentTime = args.currentTime;
				
				// 100% means 1Hz; < 1% means off; 1% is 60Hz.
				if (smoothingRate < 1) {
					smoothingRate = 0.0;
				}
				smoothingRateHz = smoothingRate ? ((1 - (((smoothingRate - 1) / 99))) * 59 + 1) : 0.0;
				//console.log("smoothingRateHz = " + smoothingRateHz);
				
				if (self.lastPauseKeyState !== pauseKeyDown) {
					var fadeDuration = (smoothingRateHz ? 3 / smoothingRateHz : 0.25),  // Make this a parameter?
						resetTransitionTimesB = true;

					// Pause state changed.
					if (pauseKeyDown) {
						var	inTransitionB = currentTime > self.transitionStartTime && currentTime < self.transitionEndTime;
						
						if (!inTransitionB) {
							fadeDuration = (smoothingRateHz ? 1 / smoothingRateHz : 0.0);	
							self.currentPose0 = currentHead14;
						} else {
							self.currentPose0 = currentHead14;
							resetTransitionTimesB = false;
						}
						self.previousPose0 = self.lastValue0; 
					} else {
						self.currentPose0 = self.lastValue0;
					}
					
					if (resetTransitionTimesB) {
						self.transitionStartTime = currentTime;
						self.transitionEndTime = currentTime + fadeDuration;
					}
				}
				self.lastPauseKeyState = pauseKeyDown;

				if (smoothingRateHz) {
					head14 = self.currentTimeFaceSmoothingFilter.addCurrentSample(currentHead14, currentTime, 1 / smoothingRateHz);
				} else {
					head14 = currentHead14;
				}
				
				if (self.currentPose0 && self.previousPose0) {
					var	interpB = currentTime >= self.transitionStartTime && currentTime < self.transitionEndTime, 
						startKey = pauseKeyDown ? self.previousPose0 : self.currentPose0, endKey = pauseKeyDown ? self.currentPose0 : head14;
					
					if (interpB) {
						var interpTime = (currentTime - self.transitionStartTime) / (self.transitionEndTime - self.transitionStartTime);
						
						interpTime = mathUtils.easeInAndOut(interpTime);
						
						var interpHead14 = [];
						for (var headLabel in head14Labels) {
							if (head14Labels.hasOwnProperty(headLabel)) {
								if (canIterpolateHeadLabel(headLabel)) {
									interpHead14[headLabel] = mathUtils.lerp(startKey[headLabel], endKey[headLabel], interpTime);		
								} else {
									interpHead14[headLabel] = endKey[headLabel];
								}
							}
						}
						
						head14 = interpHead14;
					} else {
						head14 = (currentTime === self.transitionStartTime) ? startKey : endKey;
					}
				}

				self.lastValue0 = head14;
				
				for (var label in head14Labels) {
					if (head14Labels.hasOwnProperty(label)) {
						args.eventGraph.publish1D(paramOutputKey + "Filtered/Head/" + label, args.currentTime, 
												  head14[label], !canIterpolateHeadLabel(label));
					}
				}
			}
		},
		
		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			animateWithFaceTracker(self, args);
		}
		
	}; // end of object being returned
});
